/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */
package ch.ethz.icr.benchmarking.gridipd;

import uchicago.src.sim.engine.Controller;
import uchicago.src.sim.engine.SimInit;
import uchicago.src.sim.engine.SimpleModel;
import uchicago.src.sim.space.Object2DGrid;
import uchicago.src.sim.space.Object2DTorus;
import uchicago.src.sim.util.SimUtilities;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Evolutionary game based on grid with batch capacity
 * <p/>
 * For the original model, see Cohen, Riolo, and Axelrod:
 * The Emergence of Social Organization in the Prisoner's Dilemma
 * (SFI Working Paper), 1999. http://www.santafe.edu/ (follow publications link).
 *
 * @author Nils Weidmann
 * @author Luc Girardin
 * @author Lars-Erik Cederman
 * @author Laszlo Gulyas
 */

public class Model extends SimpleModel {

    // Constants for agent strategies
    static final int ALLC = 0;
    static final int TFT = 1;
    static final int ATFT = 2;
    static final int ALLD = 3;

    // Number of possible strategies (used when looping through all possible
    // strategies).
    private final int NUMTYPES = 4;

    // Model variables
    Object2DGrid world;                   // The 2D grid (torus) representing
                                          // the world
    int worldSize;                        // The size of the world's 'sides'
    int numPlayers;                       // The number of agents
    double pALLC, pTFT, pATFT, pALLD;     // Probability of types
    int[] num;                            // Number of agents of the different
                                          // types

    private int maxIter;                  // Number of iteration cycles
    private int numNeighbors;             // The number of opponents
    double pAdapt;                        // Probability of adaptation
                                          // Note that this feature is an
                                          // extension of the original Cohen et al.
                                          // model. pAdapt=1.0 yields the original
                                          // behavior.

    double averagePayoff;                 // Average payoff of all the players

    // Constants for world types
    static final int TOPOLOGY_GRID = 0;
    static final int TOPOLOGY_TORUS = 1;
    static final int TOPOLOGY_SOUP = 2;

    // Whether this simulation is spatial or soup-like
    int topology;

    static final int NEIGHBORHOOD_VON_NEUMANN = 0;
    static final int NEIGHBORHOOD_MOORE = 1;

    // The neighborhood type switch
    int neighborhood;

    boolean showAveragePayoff;

    public Model() {
        super();
        // Setting the order of appearance of parameters
        // to NON-alphabetic
        Controller.ALPHA_ORDER = false;

        // Suppress output to the RePast console
        Controller.CONSOLE_ERR = false;
        Controller.CONSOLE_OUT = false;
    }

    /**
     * Initializing the model.
     */
    public void setup() {
        // The setup() method of the parent MUST be called first!
        super.setup();

        // Setting the name of our model
        name = "IPD model with grid topology";

        // Setting the values of the model parameters
        // Note that with the presence of the GUI, these are only
        // _default_ values.
        worldSize = 16;
        pALLC = 0.25;
        pTFT = 0.25;
        pATFT = 0.25;
        pALLD = 0.25;

        maxIter = 4;
        numNeighbors = 4;
        pAdapt = 1.0;

        topology = TOPOLOGY_TORUS;
        neighborhood = NEIGHBORHOOD_VON_NEUMANN;
    }

/////////////////////////////////////////////////////////////////////////

    // The method to build the Model's internals //////////////////////////
    public void buildModel() {

        // Creating the world
        switch(topology) {
            case TOPOLOGY_GRID:
                world = new Object2DGrid(worldSize, worldSize);
                break;
            case TOPOLOGY_TORUS:
                world = new Object2DTorus(worldSize, worldSize);
                break;
        }

        // Calculating the number of players (one player per grid cell)
        numPlayers = worldSize * worldSize;

        // Determining the number of agents in each types
        num = new int[4];   // Create the array holding the number of players per type

        // Treat each proportion as a weight on the entire sum
        double weightedNumPlayers = numPlayers / (pALLC + pTFT + pATFT + pALLD);
        num[ALLC] = (int) (weightedNumPlayers * pALLC);
        num[TFT] = (int) (weightedNumPlayers * pTFT);
        num[ATFT] = (int) (weightedNumPlayers * pATFT);
        num[ALLD] = (int) (weightedNumPlayers * pALLD);

        // Making sure we avoid cases when truncation leads to less than
        // numPlayers agents.
        if (num[ALLD] != numPlayers - num[ALLC] - num[TFT] - num[ATFT])
            num[ALLD] = numPlayers - num[ALLC] - num[TFT] - num[ATFT];

        // Creating the agents
        int playerID = 0;   // The ID of the agent under creation

        // We double-loop through each strategy and number of players
        // per strategy.
        // First, loop through all the different types
        for (int playerType = ALLC; playerType < NUMTYPES; playerType++) {
                // Creating the appropriate number of agents
            for (int i = 0; i < num[playerType]; i++) {
                // Increment the identity counter
                playerID++;

                // Create a player
                final Player aPlayer = new Player(playerID, playerType, this);

                // Add the newly created agent to the list of agents
                agentList.add(aPlayer);
            }
        }

        // Use Repast's shuffle method to randomize the order of the agents
        SimUtilities.shuffle(agentList);


        // Now it's time to put the players into the grid.
        // Since the list has been shuffled, we can scan through the array from
        // the upper left with a double-loop.
        for (int x = 0; x < worldSize; x++)
            for (int y = 0; y < worldSize; y++) {
                final Player aPlayer = (Player) agentList.get(x * worldSize + y);
                world.putObjectAt(x, y, aPlayer); // This call places the agent
                aPlayer.placeTo(x, y); // The player has to record its own position
            }

        // Show the initial statistics
        reportResults();
    }

    /////////////////////////////////////////////////////////////////////////
    //  Iterated methods  ///////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    /**
     * The main activity of the time step.
     */
    public void step() {
        // We carry out four sub-activities:
        resetPlayers();     // Reset the agents' statistics
        interactions();     // Let them interact with each other
        adaptation();       // Let them adapt
        computeResults();   // Calculate some statistics
        reportResults();    // Report some statistics
    }

    /**
     * Sub-activity: Resetting the agents.
     */
    private void resetPlayers() {
        // Loop through the entire agent list
        for (int i = 0; i < numPlayers; i++) {
            final Player aPlayer = (Player) agentList.get(i); // Pick an agent
            aPlayer.reset();                            // Let it reset it's stats
        }
    }

    /**
     * Sub-activity: Interactions.
     * The interaction method lets each player play against as many as
     * numNeighbors others
     */
    private void interactions() {
        // Looping through all the agents
        for (int i = 0; i < numPlayers; i++) {
            // Selecting the next player
            final Player aPlayer = (Player) agentList.get(i); // Get one player

            // Retrieve its neighbors (according to the neighborhood type)
            List neighbors = null;
            switch(topology) {
                case TOPOLOGY_SOUP:
                    neighbors = new ArrayList(numNeighbors);
                    for (int j = 0; j < numNeighbors; j++) {
                        int randomIndex;
                        Player other;

                        do {
                            randomIndex = getNextIntFromTo(0, numPlayers - 1);
                            other = (Player) agentList.get(randomIndex);
                        } while (other.equals(aPlayer));

                        neighbors.add(other);
                    }
                    break;
                default:
                    switch(neighborhood) {
                        case NEIGHBORHOOD_VON_NEUMANN:
                            neighbors = world.getVonNeumannNeighbors(aPlayer.x, aPlayer.y, false);
                            break;
                        case NEIGHBORHOOD_MOORE:
                            neighbors = world.getMooreNeighbors(aPlayer.x, aPlayer.y, false);
                            break;
                    }
                    break;
            }

            // Loop through them
            final Iterator it = neighbors.iterator();
            while (it.hasNext()) {
                // Pick the next neighbor...
                final Player otherPlayer = (Player) it.next();
                // ... and play a game against it.
                game(aPlayer, otherPlayer);
            }
        }
    }

    /**
     * A two-person game.
     *
     * @param rowPlayer the first player
     * @param colPlayer the second player
     */
    private void game(final Player rowPlayer, final Player colPlayer) {
        // First we need to tell the two players about each other's existence and
        // add the other player to their neighborhood memory (i.e. otherList).
        // The latter will be used in the adaptation method.
        rowPlayer.other = colPlayer;
        rowPlayer.otherList.add(colPlayer);
        colPlayer.other = rowPlayer;
        colPlayer.otherList.add(rowPlayer);

        // Here the iterated game unfolds in maxIter rounds as in SimpleIPD
        for (int i = 1; i <= maxIter; i++) {    // Note that 'time' starts at 1.
            rowPlayer.play(i);
            colPlayer.play(i);
            rowPlayer.remember();
            colPlayer.remember();
            rowPlayer.addPayoff();
            colPlayer.addPayoff();
        }
    }

    /**
     * Sub-activity: Let the agents adapt themselves to the best strategy
     * in their neighborhood (what they have recorded).
     */
    private void adaptation() {
        // NOTE DOUBLE BUFFERING!
        // Let all agents calculate their adapted type first
        for (int i = 0; i < numPlayers; i++) {
            final Player aPlayer = (Player) agentList.get(i);
            aPlayer.adapt();
        }
        // Second, once they have all calculated their new strategy,
        // let them update to the new type
        for (int i = 0; i < numPlayers; i++) {
            final Player aPlayer = (Player) agentList.get(i);
            aPlayer.updateType();
        }
    }

    /**
     * Sub-activity: Calculating the statistics.
     */
    public void computeResults() {
        averagePayoff = 0.0;

        // Initializing the temporary variables (the array of counters)
        for (int i = ALLC; i < NUMTYPES; i++)
            num[i] = 0;

        // Counting the number of players per strategy type by incrementing
        // the appropriate counter
        for (int i = 0; i < numPlayers; i++) {
            final Player aPlayer = (Player) agentList.get(i);
            averagePayoff = averagePayoff + aPlayer.getAveragePayoff();
            num[aPlayer.type]++;
        }
        averagePayoff = averagePayoff / (double) numPlayers;
    }

    /**
     * Sub-activity: Outputting the statistics.
     */
    public void reportResults() {
        // Printing the results to the screen
        System.out.print(getTickCount() + ": ");
        for (int playerType = ALLC; playerType < NUMTYPES; playerType++)
            System.out.print(num[playerType] + " ");
        System.out.println();
    }

    /**
     * Creating and starting your model.
     */
    public static void main(final String[] args) {
        final SimInit init = new SimInit();
        final Model m = new Model();
        init.loadModel(m, null, false);
    }
}