/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */
package ch.ethz.icr.benchmarking.gridipd;

import uchicago.src.sim.analysis.LocalDataRecorder;
import uchicago.src.sim.engine.SimInit;

import java.util.Date;

/**
 * Evolutionary game based on grid with batch capacity
 * <p/>
 * Running this class suppresses the GUI, and reads the parameters from a file and
 * records the results to another file.
 * <p/>
 * For the original model, see Cohen, Riolo, and Axelrod:
 * The Emergence of Social Organization in the Prisoner's Dilemma
 * (SFI Working Paper), 1999. http://www.santafe.edu/ (follow publications link).
 *
 * @author Nils Weidmann
 * @author Luc Girardin
 * @author Lars-Erik Cederman
 * @author Laszlo Gulyas
 */

public final class ModelBatch extends Model {

    // Batch variables
    private int numOfTimeSteps;     // Defines the length of each batch replication run.
    private LocalDataRecorder recorder;  // Repast's mechanism for data recording.
    private long startTime;
	private long endTime;

    /**
     * The batch module's default variable settings have to be defined.
     * Note that all the other default settings are defined in Model.setup().
     * Also note that all settings here can be overridden by the parameter file.
     */
    public final void setup() {

	    // register start time
	    startTime = (new Date()).getTime();

        // Initializing the original model first
        super.setup();

        // Specify the parameters to be manipulated by RePast's parameter
        // mechanism (i.e. set from the parameter file).
        params = new String[]{"Topology", "Neighborhood",
                              "NumOfTimeSteps", "RngSeed", "WorldSize",
                              "PALLC", "PTFT", "PATFT", "PALLD", "PAdapt"};

        // Initializing the batch part
        numOfTimeSteps = 2500;  // This sets the number of iterations.
    }

    /**
     * The method to build the Model's internals
     * Here we build the model instrumentation after the model itself has
     * been built. We make sure that the data recorder is initialized.
     */
    public final void buildModel() {
        // Set the number of simulation steps
        setStoppingTime(numOfTimeSteps);

        // Build the Batch-part
        // Create the DataRecorder object and specify the name of the output file.
        // (For details see Repast's appropriate "How to" document.)
        // In the file header, we output which JDK was used.
        recorder = new LocalDataRecorder("results_gridipd.txt", this, "JDK: "+getJDKInfo());

        // Now we add four columns in our output file. The strings that appear as
        // the last arguments refer to method names that appear below.
        recorder.createNumericDataSource("ALLC", this, "computeAllC");
        recorder.createNumericDataSource("TFT", this, "computeTFT");
        recorder.createNumericDataSource("ATFT", this, "computeATFT");
        recorder.createNumericDataSource("ALLD", this, "computeAllD");
        recorder.createNumericDataSource("Payoff", this, "computePayoff", 1, 4);
	    recorder.createNumericDataSource("ElapsedTime", this, "getElapsedTime");
        recorder.setDelimiter("|");

        // Build the original model last
        super.buildModel();

    }

    /////////////////////////////////////////////////////////////////////////
    //  Iterated methods ////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    /**
     * We suppress the original model's reportResults() method by overriding it.
     * We do so, in order to prevent excessive output on the screen.
     */
    public void reportResults() {
    }

    /**
     * We override the SimpleModel's atEnd() method to add functionality that
     * writes the simulation results to a file at the end of each replication run.
     */
    public final void atEnd() {
        super.atEnd();

        // We need to calculate the results first. For this, we use
        // the original model's reportResult() method.
        // Note that by explicitely calling super.reportResults() we
        // circumvent the 'suppression' above.
        super.reportResults();

	    //register end time before we output the results
	    endTime = (new Date()).getTime();

        // Record the results into the DataRecorder...
        recorder.record();

        // ... and write it into the file right away.
        recorder.write();
    }

    /////////////////////////////////////////////////////////////////////////
    // RePast Parameter Panel Methods ///////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    // In the following we provide get/set methods for all the parameters
    // we listed in params (see the setup() method)
    // Note that except for the two first methods, these declarations are
    // identical to the access methods in ModelGUI.

    public final int getNumOfTimeSteps() {
        return numOfTimeSteps;
    }

    public final void setNumOfTimeSteps(final int v) {
        numOfTimeSteps = v;
    }

    public final int getWorldSize() {
        return worldSize;
    }

    public final void setWorldSize(final int n) {
        worldSize = n;
    }

    public final double getPALLC() {
        return pALLC;
    }

    public final void setPALLC(final double p) {
        pALLC = p;
    }

    public final double getPTFT() {
        return pTFT;
    }

    public final void setPTFT(final double p) {
        pTFT = p;
    }

    public final double getPATFT() {
        return pATFT;
    }

    public final void setPATFT(final double p) {
        pATFT = p;
    }

    public final double getPALLD() {
        return pALLD;
    }

    public final void setPALLD(final double p) {
        pALLD = p;
    }

    public final double getPAdapt() {
        return pAdapt;
    }

    public final void setPAdapt(final double p) {
        pAdapt = p;
    }

    public int getTopology() {
        return topology;
    }

    public void setTopology(int topology) {
        this.topology = topology;
    }

	public int getNeighborhood() {
		return neighborhood;
	}

	public void setNeighborhood(int n) {
		this.neighborhood = n;
	}

    public final double computeAllC() {
        return (double) num[ALLC];
    }

    public final double computeTFT() {
        return (double) num[TFT];
    }

    public final double computeATFT() {
        return (double) num[ATFT];
    }

    public final double computeAllD() {
        return (double) num[ALLD];
    }

    public final double computePayoff() {
        return averagePayoff;
    }

	public final long getElapsedTime() {
		return endTime - startTime;
	}

	public String getJDKInfo() {
		return System.getProperty("java.vm.vendor")
		+ " " +System.getProperty("java.vm.name")+ " "
		+ System.getProperty("java.vm.version")+ " "
		+ System.getProperty("os.name");
	}

/////////////////////////////////////////////////////////////////////////
// Creating and starting your model /////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
    public static void main(final String[] args) {
        final SimInit init = new SimInit();
        // We MUST create a ModelBatch object instead of an instance of Model
        // or that of a ModelGUI, in order to get the Batch functionality.
        final Model m = new ModelBatch();
        // Note the differences between the parameters when loading a batch-mode
        // model (compared to that of a GUI-mode one, @see ModelGUI).
        // The second parameter specifies the name of the parameter file, while
        // the third one declares that the model is to be run in batch-mode.
        init.loadModel(m, "gridipd_params.conf", true);
    }
}