/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */
package ch.ethz.icr.benchmarking.gridipd;

import uchicago.src.reflector.ListPropertyDescriptor;
import uchicago.src.reflector.RangePropertyDescriptor;
import uchicago.src.sim.analysis.OpenSequenceGraph;
import uchicago.src.sim.analysis.Sequence;
import uchicago.src.sim.engine.SimInit;
import uchicago.src.sim.gui.DisplayConstants;
import uchicago.src.sim.gui.DisplaySurface;
import uchicago.src.sim.gui.Object2DDisplay;
import uchicago.src.sim.event.CheckBoxListener;
import uchicago.src.sim.event.SliderListener;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Hashtable;

/**
 * Evolutionary game based on grid with batch capacity
 * <p/>
 * This is the ModelGUI object that adds GUI functionality to the model.
 * <p/>
 * For the original model, see Cohen, Riolo, and Axelrod:
 * The Emergence of Social Organization in the Prisoner's Dilemma
 * (SFI Working Paper), 1999. http://www.santafe.edu/ (follow publications link).
 *
 * @author Luc Girardin
 * @author Lars-Erik Cederman
 * @author Laszlo Gulyas
 * @version 2.0
 */
public final class ModelGUI extends Model {

    // GUI variables
    private OpenSequenceGraph graph; // The dynamic graph chart (provided by Repast)
    public DisplaySurface dsurf;     // The display (grid) (provided by Repast)
    private int simulationSpeed;

    /**
     * The constructor of the GUI
     */
    public ModelGUI() {
        super();
    }

    /////////////////////////////////////////////////////////////////////////
    //  Definition of the sequences to be shown on the chart  ///////////////
    /////////////////////////////////////////////////////////////////////////

    /**
     * The sequence of the number of ALLC's
     */
    final class SeqALLC implements Sequence {
        // This is the method to be defined for Sequence
        public final double getSValue() {
            // Returns the appropriate value
            return (double) num[ALLC];
        }
    }

    /**
     * The sequence of the number of TFT's
     */
    final class SeqTFT implements Sequence {
        // This is the method to be defined for Sequence
        public final double getSValue() {
            // Returns the appropriate value
            return (double) num[TFT];
        }
    }

    /**
     * The sequence of the number of ATFT's
     */
    final class SeqATFT implements Sequence {
        // This is the method to be defined for Sequence
        public final double getSValue() {
            // Returns the appropriate value
            return (double) num[ATFT];
        }
    }

    /**
     * The sequence of the number of ALLD's
     */
    final class SeqALLD implements Sequence {
        // This is the method to be defined for Sequence
        public final double getSValue() {
            // Returns the appropriate value
            return (double) num[ALLD];
        }
    }

    /**
     * Initializing the GUI model
     */
    public final void setup() {
        // Initializing the original model first
        super.setup();

        // Setting the dimensions of each cell on the display (grid)
        DisplayConstants.CELL_WIDTH = 30;
        DisplayConstants.CELL_HEIGHT = 30;

        showAveragePayoff = false;
        simulationSpeed = 0;

        // Initializing the GUI-part
        // If there is already a graphics object from a previous run, we need
        // to delete it to clean up the screen.
        if (graph != null)
            graph.dispose();
        // If the display exists, let's dispose it!
        if (dsurf != null)
            dsurf.dispose();

        // Specify the parameters to be displayed for setting by the user
        // Note that we removed numPlayers as that's not an independent
        // parameter anymore, but it rather depends on worldSize.
        params = new String[]{"Topology", "Neighborhood", "WorldSize", "PALLC", "PTFT",
                              "PATFT", "PALLD", "PAdapt"};

        // Topology
        Hashtable topology = new Hashtable();
        topology.put(new Integer(TOPOLOGY_TORUS), "Torus");
        topology.put(new Integer(TOPOLOGY_GRID), "Grid");
        topology.put(new Integer(TOPOLOGY_SOUP), "Soup");
        ListPropertyDescriptor pdTopology = new ListPropertyDescriptor("Topology", topology);
        descriptors.put("Topology", pdTopology);

        // Neighborhood
        Hashtable neighborType = new Hashtable();
        neighborType.put(new Integer(NEIGHBORHOOD_VON_NEUMANN), "Von Neuman");
        neighborType.put(new Integer(NEIGHBORHOOD_MOORE), "Moore");
        ListPropertyDescriptor pdNeighborType = new ListPropertyDescriptor("Neighborhood", neighborType);
        descriptors.put("Neighborhood", pdNeighborType);

        final RangePropertyDescriptor pdWorldSize = new RangePropertyDescriptor("WorldSize",
                10, 100, 30);
        descriptors.put("WorldSize", pdWorldSize);

        // Weights for each type of player
        final RangePropertyDescriptor pdPALLC = new RangePropertyDescriptor("PALLC",
                0, 100, 25);
        descriptors.put("PALLC", pdPALLC);
        final RangePropertyDescriptor pdPTFT = new RangePropertyDescriptor("PTFT",
                0, 100, 25);
        descriptors.put("PTFT", pdPTFT);
        final RangePropertyDescriptor pdPATFT = new RangePropertyDescriptor("PATFT",
                0, 100, 25);
        descriptors.put("PATFT", pdPATFT);
        final RangePropertyDescriptor pdPALLD = new RangePropertyDescriptor("PALLD",
                0, 100, 25);
        descriptors.put("PALLD", pdPALLD);

        final RangePropertyDescriptor pdPAdapt = new RangePropertyDescriptor("PAdapt",
                0, 100, 25);
        descriptors.put("PAdapt", pdPAdapt);

        // Refresh the display surface
        modelManipulator.addButton("Refresh", new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                dsurf.repaint();;
            }
        });

        // Switch the display of the average payoff values
        CheckBoxListener showAvereragePayoffListener = new CheckBoxListener() {
            public void execute() {
                showAveragePayoff = isSelected;
                dsurf.repaint();
            }
        };
        modelManipulator.addCheckBox("Show average payoff",
                showAvereragePayoffListener, showAveragePayoff);

        // Adjust the simulation speed
        class SimulationSpeedListener extends SliderListener {
            public void execute() {
                simulationSpeed = value;
            }
        };
        modelManipulator.addSlider("Simulation speed", 0, 100, 10, new SimulationSpeedListener());

        // Now it's time to create the display surface.
        // (For details see RePast's appropriate "How to" document.)
        dsurf = new DisplaySurface(this, "ExperIPD Display");
        // Registering it as part of our GUI
        registerDisplaySurface("Main", dsurf);
    }

    /**
     * The method to build the Model's internals
     * Here we build the model instrumentation after the model itself has
     * been built.
     */
    public final void buildModel() {
        // Build the original model first
        super.buildModel();

        // Build the GUI part
        buildDisplay();
    }

    /**
     * Build the GUI part
     */
    private void buildDisplay() {
        // Create a sequence chart graph and set it up
        // (For details see RePast's appropriate "How to" document.)
        graph = new OpenSequenceGraph("Frequencies", this);
        graph.setXRange(0.0, 100.0);
        graph.setYRange(0.0, (double) numPlayers);
        graph.setAxisTitles("Generations", "Number");
        // Add the sequences we have defined
        graph.addSequence("ALLC", new SeqALLC());
        graph.addSequence("TFT", new SeqTFT());
        graph.addSequence("ATFT", new SeqATFT());
        graph.addSequence("ALLD", new SeqALLD());
        // Display the graph right away
        graph.display();
        // Do the first update to it
        graph.step();

        // Now create the 2D-display itself
        final Object2DDisplay display = new Object2DDisplay(world);
        // Let's link our collection of the agents to it.
        // (The display will show players from this list.)
        display.setObjectList(agentList);
        // Adding the grid display to the display we have created before
        dsurf.addDisplayableProbeable(display, "Display");
        // Letting RePast know that we have this display up
        addSimEventListener(dsurf);
        // Displaying the display right away
        dsurf.display();
    }

    /////////////////////////////////////////////////////////////////////////
    //  Iterated method  ////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    /**
     * The main activity of the time step.
     * In each time step, first execute the model step and then update the
     * graph
     */
    public final void step() {
        // Do the original model's step() method first
        super.step();

        // Do the GUI-part of it by updating the graph and the display
        graph.step();
        dsurf.updateDisplay();

        // Pause the simulation according the selected speed
        if(simulationSpeed < 100) {
            try {
                Thread.sleep(10 * (100 - simulationSpeed));
            } catch (InterruptedException e) {
            }
        }
    }

    /////////////////////////////////////////////////////////////////////////
    //  RePast Parameter Panel Methods  /////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    // In the following we provide get/set methods for all the parameters
    // we listed in params (see the setup() method)

    // Note that we removed numPlayers from here, too, as that's not
    // listed in params anymore.

    public final int getWorldSize() {
        return worldSize;
    }

    public final void setWorldSize(final int n) {
        worldSize = n;
    }

    public final int getPALLC() {
        return (int)(pALLC * 100.0);
    }

    public final void setPALLC(final int p) {
        pALLC = p / 100.0;
    }

    public final int getPTFT() {
        return (int)(pTFT * 100.0);
    }

    public final void setPTFT(final int p) {
        pTFT = p / 100.0;
    }

    public final int getPATFT() {
        return (int)(pATFT * 100.0);
    }

    public final void setPATFT(final int p) {
        pATFT = p / 100.0;
    }

    public final int getPALLD() {
        return (int)(pALLD * 100.0);
    }

    public final void setPALLD(final int p) {
        pALLD = p / 100.0;
    }

    public final int getPAdapt() {
        return (int)(pAdapt * 100.0);
    }

    public final void setPAdapt(final int p) {
        pAdapt = p / 100.0;
    }

    public int getTopology() {
        return topology;
    }

    public void setTopology(int topology) {
        this.topology = topology;
    }

    public int getNeighborhood() {
        return neighborhood;
    }

    public void setNeighborhood(int neighborhood) {
        this.neighborhood = neighborhood;
    }

    /**
     * Creating and starting your model.
     *
     * @param args the parameters to the program
     */
    public static void main(final String[] args) {
        final SimInit init = new SimInit();
        // We MUST create a ModelGUI object instead of an instance of Model
        // as in the parent class, in order to get the GUI functionality.
        final Model m = new ModelGUI();
        init.loadModel(m, null, false);
    }
}