/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */
package ch.ethz.icr.benchmarking.gridipd;

import uchicago.src.reflector.DescriptorContainer;
import uchicago.src.reflector.ListPropertyDescriptor;
import uchicago.src.sim.gui.DisplayConstants;
import uchicago.src.sim.gui.Drawable;
import uchicago.src.sim.gui.SimGraphics;
import uchicago.src.sim.util.SimUtilities;

import java.awt.*;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

/**
 * Evolutionary game based on grid
 * <p/>
 * For the original model, see Cohen, Riolo, and Axelrod:
 * The Emergence of Social Organization in the Prisoner's Dilemma
 * (SFI Working Paper), 1999. http://www.santafe.edu/ (follow publications link).
 *
 * @author Luc Girardin
 * @author Lars-Erik Cederman
 * @author Laszlo Gulyas
 * @version 2.0
 */
public final class Player implements Drawable,DescriptorContainer {
    // Constants for actions
    private static final int C = 1;          // Cooperate
    private static final int D = 0;          // Defect
    // Conversion array from action to String
    final String[] actionToString = {"D", "C"};

    // Action matrices
    // The x_PARAMS[] matrices define the behavior for each of the four strategies
    // (i.e., ALLC, TFT, ATFT, ALLD, respectively).
    private final int[] I_PARAMS = {C, C, D, D}; // First action
    private final int[] P_PARAMS = {C, C, D, D}; // Action if opponent cooperated
    private final int[] Q_PARAMS = {C, D, C, D}; // Action if opponent defected

    // The agent's internal variables
	int playerID;                            // the player's ID
    int x, y;                                // The player's location on the grid
    private final Model model;               // Reference (handle) to the model
    Player other;                            // Handle to the opponent
    int type;                                // The agent's strategy
    private int newType;                     // The agent's calculated strategy
    // during adaptation
    private final int[][] prefs = {{1, 5},   // Payoff / Preference matrix
                                   {0, 3}};  // (this one is PD)
    private int action;                      // The current action
    private int memory;                      // The opponent's last action
    private int cumulPayoff;                 // The agent's cumulated payoff
    private int numPlays;                    // Number of games played
                                             // (for statistics)

    final List otherList;                    // List of opponents

    private Hashtable descriptors;           // Set of parameter descriptors

    /**
     * Creating the agent.
     *
     * @param i the id of the player
     * @param t the strategy type
     * @param m the model
     */
    public Player(final int i, final int t, final Model m) {
        playerID = i;
        type = t;
        model = m;
        otherList = new ArrayList();
        descriptors = new Hashtable();

        Hashtable types = new Hashtable();
        types.put(new Integer(Model.ALLC), "ALLC");
        types.put(new Integer(Model.TFT), "TFT");
        types.put(new Integer(Model.ATFT), "ATFT");
        types.put(new Integer(Model.ALLD), "ALLD");
        ListPropertyDescriptor pdTypes = new ListPropertyDescriptor("Type", types);
        descriptors.put("Type", pdTypes);
    }

    // Setting the agent's position on the grid
    public final void placeTo(final int a, final int b) {
        x = a;
        y = b;
    }

    /**
     * Initializing the agent's variables (all counters have to be reset).
     */
    public final void reset() {
        numPlays = 0;
        cumulPayoff = 0;
        otherList.clear();
    }

    /**
     * Storing the opponent's last action.
     */
    public final void remember() {
        memory = other.action;
    }

    /**
     * The agent's decision-making.
     * This is the main behavioral method that uses one of the x_PARAMS[][]
     * matrices depending on whether the move is the first one or the other
     * player cooperated. The actual move is also dependent on the agent's
     * type. The move is recorded in the player's action variable.
     * <p/>
     * The play method is identical to that in SimpleIPD except that the numPlay
     * counter has to be incremented.
     *
     * @param time the current time
     */
    public final void play(final int time) {
        // Keeping track of the number of games played
        numPlays++;
        if (time == 1)
            action = I_PARAMS[type];
        else if (memory == C)
            action = P_PARAMS[type];
        else
            action = Q_PARAMS[type];
    }

    /**
     * Administering the payoff.
     * Updates the cumulative payoff by adding the payoffs from the preference
     * matrix as a function of both sides' moves.
     */
    public final void addPayoff() {
        cumulPayoff = cumulPayoff + prefs[action][other.action];
    }

    // In this pair of methods the player updates the strategy to that of the
    // most successful player encountered (based on the otherList).
    // (They are the same as those in GraphIPD, except the randomization
    // of the neighbor-list in adapt().)

    /**
     * Calculating the agent's new strategy.
     */
    public final void adapt() {
        // We use double-buffering by storing the result in newType which will
        // later be used to update type (but we can't to do that until we've gone
        // through all players):
        newType = type;

        // We use SimpleModel's uniform random number generator to draw a random
        // number and with probability pAdapt we let the player execute the body
        // of the method
        if (model.getNextDoubleFromTo(0.0, 1.0) < model.pAdapt) {

            // We make sure we are not biased by the order in which we check the
            // neighbors
            SimUtilities.shuffle(otherList);

            // This is a simple search loop going through the otherList to find
            // the best playoff among the opponents. The first candidate for the
            // best payoff player is the agent itself.
            double bestPayoff = getAveragePayoff();
            final Iterator i = otherList.iterator();
            while (i.hasNext()) {
                final Player aPlayer = (Player) i.next();
                final double payoff = aPlayer.getAveragePayoff();
                if (payoff > bestPayoff) {
                    bestPayoff = payoff;
                    // Set the new type to the best known (up to now)
                    newType = aPlayer.type;
                }
            }
        }
    }

    /**
     * Complete double-buffering by updating the strategy type to the
     * recently calculated value.
     */
    public final void updateType() {
        type = newType;
    }

    /**
     * Helper function returning the agent's average cumulative payoff.
     *
     * @return the agent's average cumulative payoff
     */
    public double getAveragePayoff() {
        if (numPlays == 0)          // Check that avoids division by zero
            return 0;               // Extreme value as an 'error message'
        else
            return (double) cumulPayoff / (double) numPlays;
    }

    /////////////////////////////////////////////////////////////////////////
    //  Functions of the GUI interface  /////////////////////////////////////
    /////////////////////////////////////////////////////////////////////////

    /**
     * Providing get methods for x in order to make them displayable
     *
     * @return the x coordinate of the player
     */
    public final int getX() {
        return x;
    }

    /**
     * Providing get methods for y in order to make them displayable
     *
     * @return the y coordinate of the player
     */
    public final int getY() {
        return y;
    }

    /**
     * The following accessor method make it possible to retrieve the type
     * in the probe window.
     *
     * @return the type of the player
     */
    public final int getType() {
        return type;
    }

    /**
     * The following accessor method make it possible to change the type
     * in the probe window.
     *
     * @param v the new type of the player
     */
    public final void setType(final int v) {
        type = v;
        if(model instanceof ModelGUI) {
            ((ModelGUI) model).dsurf.repaint();
        }
    }

    /**
     * Return the set of parameter descriptors
     *
     * @return the set of parameter descriptors
     */
    public Hashtable getParameterDescriptors() {
        return descriptors;
    }

    // Code to draw the agent on the grid display
    // Constant matrix to convert strategies into color-codes
    private static final Color[] COLOR = {Color.red, Color.blue, Color.green, Color.white};

    /**
     * Drawing the agent's 'dot' on the grid with the appropriate color
     *
     * @param g the graphic context
     */
    public final void draw(final SimGraphics g) {
        // Enable anti-aliasing to the underlying Java graphic context
        final Graphics2D g2 = g.getGraphics();
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
                RenderingHints.VALUE_TEXT_ANTIALIAS_ON);

        // Draw the background of each cell with the average payoff as greyscale value
        final double averagePayoff = this.getAveragePayoff();
        final double maxPayoff = 5;
        final int lum = 255 - (int) (255.0 * averagePayoff / maxPayoff);
        g.setDrawingParameters(DisplayConstants.CELL_WIDTH,
                DisplayConstants.CELL_HEIGHT,
                DisplayConstants.CELL_DEPTH);
        g.drawRect(new Color(lum, lum, lum));

        // Draw a centered circle with the stragegy color and the payoff value as text
        final int xTrans = g.getCellWidthScale();
        final int yTrans = g.getCellHeightScale();
        g.setDrawingCoordinates(getX() * xTrans + (xTrans * 3 / 16),
                getY() * yTrans + (yTrans * 3 / 16),
                0);
        g.setDrawingParameters(DisplayConstants.CELL_WIDTH * 3 / 4,
                DisplayConstants.CELL_HEIGHT * 3 / 4,
                DisplayConstants.CELL_DEPTH * 3 / 4);
        Color c = Color.white;
        if (COLOR[type] == Color.white)
            c = Color.black;
        DecimalFormat df = new DecimalFormat("#.00");

        if (model.showAveragePayoff) {
            g.drawStringInOval(COLOR[type], c, df.format(averagePayoff));
        } else {
            g.drawOval(COLOR[type]);
        }
        g.drawHollowOval(c);

        // Put back anti-aliasing to its default value
        g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.
                VALUE_TEXT_ANTIALIAS_DEFAULT);
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.
                VALUE_ANTIALIAS_DEFAULT);
    }
}