/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */

package ch.ethz.icr.benchmarking.schelling;


/**
 * Implementation of Schelling's Segregation model
 * See Schelling, Thomas C. Micromotives and Macrobehavior. New York: Norton, 1978.
 *
 * @author Nils Weidmann
 */
public abstract class Agent {

	//gives the minimum number of agents of the same color to be satisfied
	private final int[] limits = {0,1,1,2,2,2,3,3,3};

	//color constants
	public static final int WHITE = 0;
	public static final int BLACK = 1;

	public int color; //the agent's color
	protected int agentID; //the agent's ID
	protected Position position; //the agent's position
	protected Model model;  //the model the agent belongs to

	//constructor
	public Agent(Model m, int id, int c, Position p) {
		position = p;
		model = m;
		color = c;
		agentID = id;
	}

	//returns true if the agent is not satisfied at a given position
	protected boolean isDiscontent(Position pos) {

		//we ask the grid about the number and color of our neighbors
		int[] neighborStats = model.grid.getNeighborStats(pos);
		int numNeighbors = neighborStats[WHITE] + neighborStats[BLACK];

		//we use limits[] to see if we're satisfied
		boolean isDiscontent = false;
		if (neighborStats[color] < limits[numNeighbors])
			isDiscontent = true;
		return isDiscontent;
	}

	//executes the agent's move
	//Note: This method is declared abstract and must be implemented in classes extending Agent
	public abstract void move();
}
