/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */
 
package ch.ethz.icr.benchmarking.schelling;

import java.util.ArrayList;

/**
 * Implementation of Schelling's Segregation model
 * See Schelling, Thomas C. Micromotives and Macrobehavior. New York: Norton, 1978.
 *
 * @author Nils Weidmann
 */
public class Grid {

	private Agent[][] grid;  //a 2-dimensional array to store the agents
	private int dimension; //the side length of the quadratic grid

	//constructor
	public Grid(int d) {
		dimension = d;

		//create the agent array which is initialized to null values
		grid = new Agent[d][d];
	}

	//inserts an agent at a given position.
	public void putAgentAt(Agent agent, Position p) {
		if (p.x < 0 || p.x > dimension-1) return;
		if (p.y < 0 || p.y > dimension-1) return;
		grid[p.x][p.y] = agent;
	}

	//removes the agent at the given position
	public void removeAgentAt(Position p) {
		grid[p.x][p.y] = null;
	}

	//count neighbors of a given position
	public int[] getNeighborStats(Position p) {
		int[] stats = {0,0};
		for (int i=Math.max(0,p.x-1); i<=Math.min(dimension-1, p.x+1); i++) {
			for (int j=Math.max(0,p.y-1); j<=Math.min(dimension-1, p.y+1); j++) {
				if (grid[i][j] != null) {
					stats[grid[i][j].color]++;
				}
			}
		}
		return stats;
	}

	//returns a list of unoccupied positions
	public ArrayList getFreePositions() {
		ArrayList list = new ArrayList();
		for (int i=0; i<dimension; i++) {
			for (int j=0; j<dimension; j++) {
				if (grid[i][j] == null) {
					list.add(new Position(i,j));
				}
			}
		}
		return list;
	}

	//returns a String representation of the current configuration
	public String toString() {

		StringBuffer outputString = new StringBuffer();

		for (int j=0; j<dimension+2; j++) {
			outputString.append("-");
		}
		outputString.append("\n");

		for (int j=0; j<dimension; j++) {
			outputString.append("|");
			for (int i=0; i<dimension; i++) {
				if (grid[i][j] == null) {
					outputString.append(" ");
				}
				else if(grid[i][j].color == Agent.WHITE) {
					outputString.append("O");
				}
				else outputString.append("#");
			}
			outputString.append("|\n");
		}
		for (int j=0; j<dimension+2; j++) {
			outputString.append("-");
		}
		outputString.append("\n");
		return outputString.toString();
	}
}

