/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */

package ch.ethz.icr.benchmarking.schelling;

import java.util.Random;
import java.util.ArrayList;
import java.util.Date;
import java.io.FileWriter;
import java.io.File;

/**
 * Implementation of Schelling's Segregation model
 * See Schelling, Thomas C. Micromotives and Macrobehavior. New York: Norton, 1978.
 *
 * @author Nils Weidmann
 */
public class Model {

	//model parameters
	private static int gridDimension = 100; //dimension of the square grid
	private double populationRatio = 0.9; //between 0 and 1
	private double colorRatio = 0.5; //probability for agents of being black, and white otherwise
	private int seed; //the random seed
	private int iterations = 100; //the number of iterations to perform
	private static int maxSeed = 30;  //the number of seeds to test in a batch run
	public boolean printResults = false; //print moves and grid to console?

	//class variables
	public Grid grid; //the grid storing the agents
	private ArrayList agents; //the model's agent list
	public Random random; //the model's random number generator


	public static void main(String[] args) {

		// we try to mimick RePast's behavior with output files
		try {
			//fix a file name
			String midEx = "";
			int count = 0;
			File outFile;
			do {
				outFile = new File("results_schelling"+midEx+".txt");
				count++;
				midEx = ".bak"+count;
			}
			while(outFile.exists());

			//open file and write header info
			FileWriter out = new FileWriter(outFile);
			out.write("JDK: "
			        + System.getProperty("java.vm.vendor") + " "
			        + System.getProperty("java.vm.name")+ " "
			        + System.getProperty("java.vm.version")+ " "
			        + System.getProperty("os.name"));
			out.write("\n\n");
			out.write("gridDimension|populationRatio|colorRatio|seed|elapsedTime\n");

			//main loop with different seeds
			for (int s = 1; s <= maxSeed; s++) {

				// we register the start time
				long startTime = (new Date().getTime());

				// execute the model
				Model model = new Model(s);
				model.initModel();
				model.runModel();

				// compute the elapsed time
				long elapsedTime = (new Date().getTime()) - startTime;

				// and write everything to the output file
				out.write(model.gridDimension+"|"+model.populationRatio+"|"+model.colorRatio+"|"+s+"|"+elapsedTime+"\n");
				System.out.println("Completed run with random seed = "+s+" and worldSize = "+gridDimension + ".");
			}

			// close the output stream
			out.close();
		}
		catch (Exception e) {};
	}

	public Model(int s) {
		seed = s;
	}


	//creates the grid and populates it with new agents
	private void initModel() {
		grid = new Grid(gridDimension);
		agents = new ArrayList();
		random = new Random(seed);
		int agentID = -1;

		//we go through the 2-dimensional grid
		for (int i=0; i<gridDimension; i++) {
			for (int j=0; j<gridDimension; j++) {

				//with probability populationRatio we create an agent
				if(random.nextDouble()<populationRatio) {

					//we probability colorRatio, the agent's color is black
					int agentColor = PickyAgent.WHITE;
					if (random.nextDouble() < colorRatio)
						agentColor = PickyAgent.BLACK;
					agentID++;
					Position agentPosition = new Position(i,j);

					//here we define the type of agent we want to use
					//has to be changed if you want to use your own agent class
					Agent newAgent = new PickyAgent(this, agentID, agentColor, agentPosition);

					//put the agent in the grid and in the model's agent list
					grid.putAgentAt(newAgent, agentPosition);

					agents.add(newAgent);
				}
			}
		}
	}

	//runs the model iterations
	private void runModel() {
		if (printResults) {
			System.out.println("Initial configuration:");
			System.out.println(grid);  //calls simpleschelling.Grid.toString()
		}
		for (int s=0; s<iterations; s++) {

			//in each step, we n-times pick a random agent and let it move, where n is the number of agents
			for (int a=0; a<agents.size(); a++) {
				int index = random.nextInt(agents.size());
				Agent agent = (Agent) agents.get(index);
				agent.move();
			}
			if (printResults) {
				System.out.println("\nConfiguration after step "+s+":");
				System.out.println(grid);
			}
		}
	}
}
