/*
 * Copyright (c) 2004 International Conflict Research
 * at the Swiss Federal Institute of Technology Zurich
 * (see http://www.icr.ethz.ch/ for details)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Please see http://www.gnu.org/copyleft/gpl.txt for the full license text.
 *
 */

package ch.ethz.icr.benchmarking.schelling;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * Implementation of Schelling's Segregation model
 * See Schelling, Thomas C. Micromotives and Macrobehavior. New York: Norton, 1978.
 *
 * @author Nils Weidmann
 */

public class PickyAgent extends Agent {

	public PickyAgent(Model m, int id, int c, Position p) {
		super(m,id,c,p);
	}

	//executes the agent's move
	public void move() {

		//in order not to mess up the stats, we remove ourselves from the grid
		model.grid.removeAgentAt(position);

		//see if we're content at our current position
		if (isDiscontent(position)) {

			//ask the grid about empty positions
			ArrayList freePositions = model.grid.getFreePositions();

			//remove current position and positions where we're not satisfied,
			//as we do not want to move to those positions
			Iterator it = freePositions.iterator();
			Position pos;
			while (it.hasNext()) {
				pos = (Position) it.next();
				if ((pos.x == position.x && pos.y == position.y) || isDiscontent(pos))
					it.remove();
			}

			//if there are convenient spaces for us, we randomly pick one...
			if (freePositions.size() > 0) {

				int randIndex = model.random.nextInt(freePositions.size());
				Position newPos = (Position) freePositions.get(randIndex);

				//...we print where we want to move...
				if (model.printResults)
					System.out.println("Agent "+agentID+" moves from ("+position.x+","+position.y+") to ("+newPos.x+","+newPos.y+").");

				//... and move to the new position
				position = newPos;
			}
		}

		//put ourselves back on the grid
		model.grid.putAgentAt(this, position);
	}
}
